import json
import logging

from cse_image_items.utils import set_item_data
from utils import bytes_to_int


logger = logging.getLogger(__name__)

class BaseItem(object):
    """Base class for all items
    """

    def __init__(self, data=None, **kwargs):
        """
        Arguments:
            data {bytes} -- Item bytes from CSE FW image
            kwargs {dict} -- Describes the data_bytes
        """
        members = {}
        members.update(kwargs)
        self.offset, self.size, self.value = set_item_data(data, members)

    def bytes_value(self, byteorder='little'):
        """Getter for item bytes

        Keyword Arguments:
            byteorder {str} -- byteorder (default: {'little'})

        Returns:
            bytes -- item bytes
        """
        try:
            assert byteorder == 'little' or byteorder == 'big'
        except AssertionError:
            raise AssertionError('byteorder [%s] must be little or big' % (byteorder))
            logger.exception('byteorder [%s] must be little or big' % (byteorder))
        return self.value if byteorder == 'little' else self.value[::-1]

    def int_value(self):
        """Getter for item bytes as integer

        Returns:
            int -- item bytes as integer
        """
        return bytes_to_int(self.bytes_value())

    def hex_value(self):
        """Getter for item bytes as hex string

        Returns:
            str -- item bytes as hex string
        """
        return hex(self.int_value()).decode()

    def bool_value(self):
        """Getter for item bytes as boolean

        Returns:
            bool -- item bytes as boolean
        """
        return self.int_value() != 0

    def str_value(self):
        """Getter for item bytes as string

        Returns:
            str -- item bytes as string
        """
        return self.bytes_value().decode('unicode_escape').rstrip('\x00')
